/** @file   tprinter.cpp
 * @brief   Implementation of TPrinter class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "tprinter.h"

using std::string;


namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================

/** Prints the given string
 */
void TPrinter::print( BITMAP* pB, const GlyphSet* pF, const string& rTxt,
                      int* pX, int* pY )
{
  if ( pF == 0 ) {
    return;
  }
  
  int posX = *pX;
  int posY = *pY;
  for ( int i=0; i < rTxt.length(); i++ ) {
    int c = rTxt.at(i);
    BITMAP* pGfx = pF->gfx( c );
    if ( pGfx != 0 && pB != 0 ) {
      // Draw the glyph.
      draw_sprite( pB, pGfx, posX, posY );
    }
    
    // Move the printing position.
    posX += pF->glyphWidth( c );
  }
  
  *pX = posX;
  *pY = posY;
}


/** The conversion buffer */
char TPrinter::s_convertBuffer[33];


///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
TPrinter::TPrinter() :
  m_pTarget( 0 ),
  m_pF( 0 ),
  m_x( 0 ),
  m_y( 0 ),
  m_align( ALIGN_LEFT )
{
}



/** Constructor
 */
TPrinter::TPrinter( BITMAP* pB ) :
  m_pTarget( pB ),
  m_pF( 0 ),
  m_x( 0 ),
  m_y( 0 ),
  m_align( ALIGN_LEFT )
{
}



/** Destructor
 */
TPrinter::~TPrinter()
{
}




///
/// Public methods
/// ==============

/** Prints the given string
 */
void TPrinter::print( const string& txt )
{
  switch ( m_align ) {
    case ( ALIGN_LEFT ): {
      TPrinter::print( m_pTarget, m_pF, txt, &m_x, &m_y );
      break;
    }
    case ( ALIGN_CENTER ): {
      int tlen = this->textLength( txt );
      int tmpX = m_x - (tlen / 2 );
      TPrinter::print( m_pTarget, m_pF, txt, &tmpX, &m_y );
      m_x += tlen / 2;
      break;
    }
    case ( ALIGN_RIGHT ): {
      int tlen = this->textLength( txt );
      int tmpX = m_x - tlen;
      TPrinter::print( m_pTarget, m_pF, txt, &tmpX, &m_y );
      break;
    }
  }
}



/** Prints the given number
 */
void TPrinter::print( int number )
{
  string txt( itoa( number, s_convertBuffer, 10 ) );
  this->print( txt );
}





/** Sets the position
 */
void TPrinter::pos( int x, int y )
{
  m_x = x;
  m_y = y;
}



/** Sets the bitmap
 */
void TPrinter::bitmap( BITMAP* pB )
{
  m_pTarget = pB;
}



/** Sets the font
 */
void TPrinter::font( const BMFont* pF , int siz )
{
  if ( pF == 0 ) {
    return;
  }
  m_pF = pF->glyphset( siz );
}



/** Sets the font
 */
void TPrinter::font( const BMFont* pF, const string& id )
{
  if ( pF == 0 ) {
    return;
  }
  m_pF = pF->glyphset( id );
}



/** The aligning
 */
void TPrinter::align( Align a )
{
  m_align = a;
}



///
/// Getter methods
/// ==============

/** Returns the length of the text in pixels
 */
int TPrinter::textLength( const string& txt )
{
  int tmpX = 0;
  int tmpY = 0;
  TPrinter::print( 0, m_pF, txt, &tmpX, &tmpY );
  return tmpX;
}



/** Returns the length of the integer in pixels
 */
int TPrinter::textLength( int number )
{
  int tmpX = 0;
  int tmpY = 0;
  
  string txt( itoa( number, s_convertBuffer, 10 ) );
  TPrinter::print( 0, m_pF, txt, &tmpX, &tmpY );
}




///
/// Private or Protected methods
/// ============================

} // end of namespace
